function [M labels] = functionalCartography(obj, varargin)
%FUNCTIONALCARTOGRAPHY Calculate functional cartography of AdjacencyMatrix
%object
%   [M labels] = FUNCTIONALCARTOGRAPHY(obj) will calculate the functional
%   cartography matrix of AdjacencyMatrix obj, based on the method outlined
%   in Guimera and Amaral, Functional cartography of complex metabolic
%   networks, Nature, Feb. 2005.  M will be a N x 2 matrix, where N is the
%   number of neurons, and labels will give the neuron names.  The first
%   column of N will be the within-module z-score, and the second column
%   will be the participation coefficient

[commLabels Q] = getCommunitiesMM(obj, 'quiet');

% First column of data is within-module degree
% Second column of data is within-module z-score
% Third column of data is participation coefficient
data = zeros(numel(obj.cells),3);
labels = {};

% Calculate within-module degree z-score
k = 1;
for i = 1:numel(commLabels)
    [currModule Amod] = getSubMatrix(obj,commLabels{i}',commLabels{i});
    labelsMod = commLabels{i};
    %     [Amod labelsMod] = getSquareMatrix(currModule);
    degsInMod = sum(Amod,1)' + sum(Amod,2);
    aveDegsInMod = mean(degsInMod);
    sigDegsInMod = std(degsInMod);
    data(k:(k+numel(labelsMod)-1),1) = degsInMod;
    data(k:(k+numel(labelsMod)-1),2) = (degsInMod-aveDegsInMod)./sigDegsInMod;
    labels(k:(k+numel(labelsMod)-1)) = labelsMod;
    k = k + numel(labelsMod);
end

% Calculate participation coefficient
% newA will have columns and rows in order of modules
[s newA] = getSubMatrix(obj,labels);
degs = sum(newA)' + sum(newA,2);
k = 1;
for i = 1:numel(commLabels)
    [currModule Amod] = getSubMatrix(obj,commLabels{i}',commLabels{i});
    
    data(:,3) = data(:,3) + ((sum(newA(:,k:(k+numel(currModule.cells)-1)),2)+ ... 
        sum(newA(k:(k+numel(currModule.cells)-1),:))')./degs).^2;
    k = k + numel(currModule.cells);
end

data(:,3) = 1-data(:,3);
M = [data(:,2) data(:,3)];

figure;
scatter(M(:,2),M(:,1));
hold on
text(M(:,2),M(:,1),labels);
xlabel('Participation coefficient, P');
ylabel('Within-module degree, z');
title('Functional cartography');
set(gcf,'Color',[1 1 1]);

% Print out node types
disp('Ultra-peripheral nodes');
fprintf('%s\n', labels{M(:,1)<2.5 & M(:,2)<.05})
fprintf('\n')

disp('Peripheral nodes');
fprintf('%s\n', labels{M(:,1)<2.5 & M(:,2)>.05 & M(:,2)<.62})
fprintf('\n')

disp('Non-hub connector nodes');
fprintf('%s\n', labels{M(:,1)<2.5 & M(:,2)>.62 & M(:,2)<.8})
fprintf('\n')

disp('Non-hub kinless nodes');
fprintf('%s\n', labels{M(:,1)<2.5 & M(:,2)>.80})
fprintf('\n')


disp('Provincial hubs');
fprintf('%s\n', labels{M(:,1)>2.5 & M(:,2)<.30})
fprintf('\n')

disp('Connector hubs');
fprintf('%s\n', labels{M(:,1)>2.5 & M(:,2)>.30 & M(:,2)<.75})
fprintf('\n')

disp('Kinless hubs');
fprintf('%s\n', labels{M(:,1)>2.5 & M(:,2)>.75})
fprintf('\n')


end

